<?php
// add_product.php - Handles adding new products to products.json

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Function to sanitize input
function sanitizeInput($input) {
    return htmlspecialchars(stripslashes(trim($input)), ENT_QUOTES, 'UTF-8');
}

// Function to create slug from title
function createSlug($title) {
    $slug = strtolower($title);
    $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
    $slug = preg_replace('/[\s-]+/', '-', $slug);
    $slug = trim($slug, '-');
    return $slug;
}

// Function to calculate discount percentage
function calculateDiscount($price, $retailValue) {
    if ($retailValue > 0) {
        $discount = (($retailValue - $price) / $retailValue) * 100;
        return round($discount, 1) . '% Off';
    }
    return '0% Off';
}

// Get form data
$title = sanitizeInput($_POST['product-name'] ?? '');
$id = sanitizeInput($_POST['product-id'] ?? '');
$category = sanitizeInput($_POST['product-category'] ?? '');
$price = floatval($_POST['product-price'] ?? 0);
$retailValue = floatval($_POST['product-retail-value'] ?? 0);
$quantity = intval($_POST['product-quantity'] ?? 0);
$costPerUnit = floatval($_POST['product-cost-per-unit'] ?? 0);
$quality = sanitizeInput($_POST['product-quality'] ?? '');
$manifestStatus = sanitizeInput($_POST['product-manifest-status'] ?? '');
$status = sanitizeInput($_POST['status'] ?? 'Published');

// Validate required fields
if (empty($title) || empty($id) || empty($category) || $price <= 0 || $retailValue <= 0 || $quantity <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'All required fields must be filled']);
    exit;
}

// Create slug
$slug = createSlug($title);

// Calculate discount
$discount = calculateDiscount($price, $retailValue);

// Handle image uploads
$imageUrls = [];
$uploadDir = __DIR__ . '/uploads/products/';

if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

if (isset($_FILES['product-image-file'])) {
    $files = $_FILES['product-image-file'];
    
    // Handle multiple files
    if (is_array($files['name'])) {
        for ($i = 0; $i < count($files['name']); $i++) {
            if ($files['error'][$i] === UPLOAD_ERR_OK) {
                $fileName = basename($files['name'][$i]);
                $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                $allowedExts = ['jpg', 'jpeg', 'png', 'webp'];
                
                if (in_array($fileExt, $allowedExts)) {
                    $newFileName = uniqid() . '.' . $fileExt;
                    $filePath = $uploadDir . $newFileName;
                    
                    if (move_uploaded_file($files['tmp_name'][$i], $filePath)) {
                        $imageUrls[] = 'uploads/products/' . $newFileName;
                    }
                }
            }
        }
    } else {
        // Single file
        if ($files['error'] === UPLOAD_ERR_OK) {
            $fileName = basename($files['name']);
            $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $allowedExts = ['jpg', 'jpeg', 'png', 'webp'];
            
            if (in_array($fileExt, $allowedExts)) {
                $newFileName = uniqid() . '.' . $fileExt;
                $filePath = $uploadDir . $newFileName;
                
                if (move_uploaded_file($files['tmp_name'], $filePath)) {
                    $imageUrls[] = 'uploads/products/' . $newFileName;
                }
            }
        }
    }
}

// If no images uploaded, use a placeholder
if (empty($imageUrls)) {
    $imageUrls[] = 'https://via.placeholder.com/800x600?text=No+Image';
}

$imageUrl = $imageUrls[0];
$imageGallery = $imageUrls;
$imageAlt = $title;

// Load existing products
$productsFile = __DIR__ . '/data/products.json';
$products = [];

if (file_exists($productsFile)) {
    $productsJson = file_get_contents($productsFile);
    $products = json_decode($productsJson, true) ?: [];
}

// Create new product
$newProduct = [
    'id' => $id,
    'title' => $title,
    'slug' => $slug,
    'category' => $category,
    'discount' => $discount,
    'imageUrl' => $imageUrl,
    'imageGallery' => $imageGallery,
    'imageAlt' => $imageAlt,
    'estRetailValue' => number_format($retailValue, 2, '.', ','),
    'manifestStatus' => $manifestStatus,
    'quality' => $quality,
    'quantity' => $quantity,
    'price' => number_format($price, 2),
    'costPerUnit' => number_format($costPerUnit, 2)
];

// Add to products array
$products[] = $newProduct;

// Save back to file
if (file_put_contents($productsFile, json_encode($products, JSON_PRETTY_PRINT))) {
    echo json_encode(['success' => true, 'message' => 'Product added successfully']);
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Failed to save product']);
}
?>